// Lawyer Consultation Website - Version 3 (HTML/CSS/JS)

const areas = [
  { 
    id: 'family', 
    name: 'Семейное право', 
    icon: 'icons/area-family.svg',
    color: '#1E40AF',
    description: 'Разводы, алименты, опека',
    duration: '1-1.5 часа',
    price: 'от 3000₽'
  },
  { 
    id: 'labor', 
    name: 'Трудовое право', 
    icon: 'icons/area-labor.svg',
    color: '#3B82F6',
    description: 'Трудовые споры, увольнения',
    duration: '1-2 часа',
    price: 'от 3500₽'
  },
  { 
    id: 'housing', 
    name: 'Жилищное право', 
    icon: 'icons/area-housing.svg',
    color: '#60A5FA',
    description: 'Сделки с недвижимостью',
    duration: '1.5-2 часа',
    price: 'от 4000₽'
  },
  { 
    id: 'criminal', 
    name: 'Уголовное право', 
    icon: 'icons/area-criminal.svg',
    color: '#2563EB',
    description: 'Защита по уголовным делам',
    duration: '2-3 часа',
    price: 'от 5000₽'
  }
]

const lawyers = [
  { name: 'Анна Соколова', experience: '15 лет', rating: 5, specialty: 'Семейное, Жилищное право', cases: '500+' },
  { name: 'Михаил Петров', experience: '12 лет', rating: 5, specialty: 'Трудовое, Корпоративное право', cases: '400+' },
  { name: 'Елена Волкова', experience: '18 лет', rating: 5, specialty: 'Уголовное, Административное право', cases: '600+' },
  { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Жилищное, Семейное право', cases: '350+' }
]

const benefits = [
  { icon: 'icons/stat-status.svg', title: 'Защита прав', text: 'Максимальная защита ваших интересов' },
  { icon: 'icons/stat-lawyer.svg', title: 'Опытные юристы', text: 'Специалисты с многолетним опытом' },
  { icon: 'icons/form-calendar.svg', title: 'Быстрая помощь', text: 'Консультации в удобное для вас время' },
  { icon: 'icons/stat-consultations.svg', title: 'Высокий рейтинг', text: '98% довольных клиентов' }
]

const guarantees = [
  { icon: 'icons/stat-status.svg', title: 'Конфиденциальность', text: 'Полная защита персональных данных' },
  { icon: 'icons/stat-lawyer.svg', title: 'Профессионализм', text: 'Только опытные специалисты' },
  { icon: 'icons/form-calendar.svg', title: 'Оперативность', text: 'Быстрые ответы на ваши вопросы' },
  { icon: 'icons/stat-consultations.svg', title: 'Результативность', text: '95% успешных решений дел' }
]

function initPage() {
  const currentPage = window.location.pathname.includes('new.html') ? 'new' : 'index'
  
  if (currentPage === 'new') {
    initNewPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadConsultations()
  updateStats()
  renderBenefits()
  
  setInterval(() => {
    loadConsultations()
    updateStats()
  }, 500)
}

function loadConsultations() {
  const saved = localStorage.getItem('lawConsultations')
  let consultations = []
  
  if (saved) {
    try {
      consultations = JSON.parse(saved)
    } catch (e) {
      consultations = []
    }
  } else {
    consultations = [
      {
        id: 1,
        topic: 'Семейное право',
        lawyer: 'Анна Соколова',
        status: 'Завершено',
        review: 'Отличная консультация, все разъяснили подробно'
      },
      {
        id: 2,
        topic: 'Трудовое право',
        lawyer: 'Михаил Петров',
        status: 'В процессе',
        review: null
      },
      {
        id: 3,
        topic: 'Жилищное право',
        lawyer: 'Елена Волкова',
        status: 'Завершено',
        review: 'Профессионал своего дела, рекомендую!'
      }
    ]
    localStorage.setItem('lawConsultations', JSON.stringify(consultations))
  }
  
  renderConsultations(consultations)
  return consultations
}

function getAreaColor(topic) {
  const colors = {
    'Семейное право': '#1E40AF',
    'Трудовое право': '#3B82F6',
    'Жилищное право': '#60A5FA',
    'Уголовное право': '#2563EB'
  }
  return colors[topic] || '#1E40AF'
}

function getStatusColor(status) {
  return status === 'Завершено' ? '#10B981' : '#F59E0B'
}

function renderConsultations(consultations) {
  const consultationsGrid = document.getElementById('consultationsGrid')
  if (!consultationsGrid) return
  
  if (consultations.length === 0) {
    consultationsGrid.innerHTML = `
      <div class="empty-state">
        <div class="empty-icon-circle">
          <img src="icons/form-scale.svg" alt="" width="72" height="72" />
        </div>
        <h3 class="empty-title">Нет обращений</h3>
        <p class="empty-text">Запишитесь на консультацию прямо сейчас</p>
      </div>
    `
  } else {
    consultationsGrid.innerHTML = consultations.map(consultation => {
      const areaColor = getAreaColor(consultation.topic)
      const statusColor = getStatusColor(consultation.status)
      
      return `
        <div class="consultation-card" style="border-left: 5px solid ${areaColor};">
          <div class="consultation-header">
            <div class="consultation-topic-badge" style="background: ${areaColor}20; color: ${areaColor};">
              ${consultation.topic}
            </div>
            <div class="consultation-status" style="background: ${statusColor}20; color: ${statusColor};">
              ${consultation.status}
            </div>
          </div>
          <div class="consultation-content">
            <div class="consultation-info-item">
              <img src="icons/stat-lawyer.svg" alt="" width="22" height="22" style="filter: drop-shadow(0 2px 4px ${areaColor}30);" />
              <span class="consultation-info-text">${consultation.lawyer}</span>
            </div>
            ${consultation.review ? `
              <div class="consultation-review">
                <img src="icons/form-scale.svg" alt="" width="20" height="20" style="filter: drop-shadow(0 2px 4px ${areaColor}30);" />
                <span class="review-text">${consultation.review}</span>
              </div>
            ` : ''}
          </div>
        </div>
      `
    }).join('')
  }
}

function updateStats() {
  const consultations = loadConsultations()
  const stats = [
    { 
      value: consultations.length, 
      label: 'Обращений',
      icon: 'icons/stat-consultations.svg',
      color: '#1E40AF'
    },
    { 
      value: consultations.filter(c => c.status === 'Завершено').length, 
      label: 'Завершено',
      icon: 'icons/stat-status.svg',
      color: '#10B981'
    },
    { 
      value: new Set(consultations.map(c => c.lawyer)).size, 
      label: 'Юристов',
      icon: 'icons/stat-lawyer.svg',
      color: '#3B82F6'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="stat-card">
        <div class="stat-icon-wrapper" style="background: ${stat.color}18;">
          <img src="${stat.icon}" alt="" width="40" height="40" />
        </div>
        <div class="stat-content">
          <div class="stat-value">${stat.value}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }
}

function renderBenefits() {
  const benefitsGrid = document.getElementById('benefitsGrid')
  if (benefitsGrid) {
    benefitsGrid.innerHTML = benefits.map(benefit => `
      <div class="benefit-item">
        <div class="benefit-icon" style="background: rgba(30, 64, 175, 0.15);">
          <img src="${benefit.icon}" alt="" width="36" height="36" />
        </div>
        <h3 class="benefit-title">${benefit.title}</h3>
        <p class="benefit-text">${benefit.text}</p>
      </div>
    `).join('')
  }
}

function initNewPage() {
  const consultationForm = document.getElementById('consultationForm')
  const areasGrid = document.getElementById('areasGrid')
  const dateInput = document.getElementById('date')
  const successMessage = document.getElementById('successMessage')
  const lawyersGrid = document.getElementById('lawyersGrid')
  const guaranteesGrid = document.getElementById('guaranteesGrid')

  let selectedArea = null

  if (areasGrid) {
    areasGrid.innerHTML = areas.map(area => `
      <button type="button" class="area-card" data-id="${area.id}" style="border-color: rgba(30, 64, 175, 0.25);">
        <div class="area-icon-wrapper" style="background: ${area.color}25;">
          <img src="${area.icon}" alt="${area.name}" width="44" height="44" style="filter: drop-shadow(0 2px 4px ${area.color}30);" />
        </div>
        <div class="area-content">
          <h3 class="area-name">${area.name}</h3>
          <p class="area-description">${area.description}</p>
          <div class="area-meta">
            <span class="area-duration">${area.duration}</span>
            <span class="area-price">${area.price}</span>
          </div>
        </div>
      </button>
    `).join('')

    areasGrid.querySelectorAll('.area-card').forEach(button => {
      button.addEventListener('click', function() {
        areasGrid.querySelectorAll('.area-card').forEach(btn => {
          btn.classList.remove('selected')
          btn.style.borderColor = 'rgba(30, 64, 175, 0.25)'
          const check = btn.querySelector('.area-check')
          if (check) check.remove()
        })
        this.classList.add('selected')
        this.style.borderColor = areas.find(a => a.id === this.dataset.id).color
        selectedArea = areas.find(a => a.id === this.dataset.id)
        
        const check = document.createElement('div')
        check.className = 'area-check'
        check.style.background = selectedArea.color + '15'
        check.innerHTML = `<img src="icons/btn-check.svg" alt="" width="28" height="28" style="filter: drop-shadow(0 2px 4px ${selectedArea.color}30);" />`
        this.appendChild(check)
      })
    })
  }

  if (lawyersGrid) {
    lawyersGrid.innerHTML = lawyers.map(lawyer => {
      const stars = Array.from({ length: 5 }, () => 
        `<img src="icons/stat-status.svg" alt="star" width="18" height="18" style="filter: drop-shadow(0 2px 4px rgba(245, 158, 11, 0.3));" />`
      ).join('')
      
      return `
        <div class="lawyer-card">
          <div class="lawyer-avatar">
            <img src="icons/stat-lawyer.svg" alt="" width="44" height="44" />
          </div>
          <h3 class="lawyer-name">${lawyer.name}</h3>
          <p class="lawyer-experience">${lawyer.experience} опыта</p>
          <div class="lawyer-rating">
            ${stars}
          </div>
          <p class="lawyer-specialty">${lawyer.specialty}</p>
        </div>
      `
    }).join('')
  }

  if (guaranteesGrid) {
    guaranteesGrid.innerHTML = guarantees.map(guarantee => `
      <div class="guarantee-card">
        <div class="guarantee-icon">
          <img src="${guarantee.icon}" alt="" width="36" height="36" />
        </div>
        <h3 class="guarantee-title">${guarantee.title}</h3>
        <p class="guarantee-text">${guarantee.text}</p>
      </div>
    `).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (consultationForm) {
    consultationForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedArea && dateInput.value) {
        const savedConsultations = localStorage.getItem('lawConsultations')
        const consultations = savedConsultations ? JSON.parse(savedConsultations) : []
        
        const randomLawyer = lawyers[Math.floor(Math.random() * lawyers.length)]
        const newConsultation = {
          id: Date.now(),
          topic: selectedArea.name,
          lawyer: randomLawyer.name,
          status: 'В процессе',
          review: null
        }

        consultations.push(newConsultation)
        localStorage.setItem('lawConsultations', JSON.stringify(consultations))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedArea = null
          dateInput.value = ''
          areasGrid.querySelectorAll('.area-card').forEach(btn => {
            btn.classList.remove('selected')
            btn.style.borderColor = 'rgba(30, 64, 175, 0.25)'
            const check = btn.querySelector('.area-check')
            if (check) check.remove()
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  slides[currentSlide].classList.remove('active')
  dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

